/* Language: C */

#include <assert.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/*
 * Output OpenSCAD code to provide tabular information for a multiplying rod 
 * which uses Slonimsky's theorem.
 *
 * David Moews, 3-VI-2014.
 *
 * This code is in the public domain.
 */

#define MAX_BASE 36
#define MAX_CARRIES (MAX_BASE*(MAX_BASE-1))/2

typedef struct
{
  int n, d;
} Rational;

int base;

/* This array holds minimal rationals for each carry-in vector, 
 * in increasing order */
Rational carries[MAX_CARRIES];
int n_carries = 0;

int gcd(int a, int b)
{
  if (a < 0) a = -a;
  if (b < 0) b = -b;
  while (b != 0)
  {
    int r = a % b;
    a = b;
    b = r;
  }
  return a;
}

int cmp_rational(const void *a, const void *b)
{
  const Rational *ra = (const Rational *)a;
  const Rational *rb = (const Rational *)b;
  return ra->n * rb->d - rb->n * ra->d;
}

int get_cout(int dig, int carry_in)
{
  Rational r;
  int j;

  /* Input carry rational n/d becomes output carry rational */
  /* (n/d + (dig))/base = (n + (dig) d)/(base d). */
  r.d = base * carries[carry_in].d;
  r.n = carries[carry_in].n + dig * carries[carry_in].d;

  /* Find interval holding output carry rational */
  j = 0;
  while (j < n_carries - 1 && cmp_rational(&r, &carries[j + 1]) >= 0)
        j++;
  assert(j < n_carries && cmp_rational(&r, &carries[j]) >= 0);
  if (j < n_carries - 1)
     assert(cmp_rational(&r, &carries[j+1]) < 0);
  return j;
}

int get_output_digit(int dig, int carry_in, int j)
{
  return (dig * j + (carries[carry_in].n * j) / carries[carry_in].d) % base;
}

void usage(const char *n)
{
  fprintf(stderr, "Outputs OpenSCAD table file for multiplying rods which use Slonimsky's theorem.\n\n");
  fprintf(stderr, "Usage: %s base digit-alph carry-alph\n", n);
  fprintf(stderr, "\nwhere\n\
\n\
 base           is the base of the number system.\n\
 digit-alph     is the string of characters to use for digits \n\
                (its length should equal the base.)\n\
 carry-alph     is the string of characters to use for carry symbols\n\
                (as many as required; 28 for base 10.)\n\
\n\
 Sample arguments:\n\
\n\
 %s 10 0123456789 ' ABCDEFGHIJKLMNOPQRSTUVWXYZ*'\n\
", n);
  return;
}

int main(int argc, char **argv)
{
  int n, d, i, j, dig;
  const char *dig_alphabet, *carry_alphabet;

  if (argc != 4)
  {
    usage(argv[0]);
    return 1;
  }
  base = atoi(argv[1]);
  dig_alphabet = argv[2];
  carry_alphabet = argv[3];
  if (base < 2 || base > MAX_BASE)
  {
    fprintf(stderr, "Base out of range (should be 2..%d.)\n", MAX_BASE);
    return 2;
  }
  if (strlen(dig_alphabet) != base)
  {
    fprintf(stderr, "Digit alphabet wrong length (should be %d.)\n", base);
    return 3;
  }

  n_carries = 0;

  for (d = 1; d < base; d++)
  for (n = 0; n < d; n++)
  if (gcd(n, d) == 1)
  {
    assert(n_carries < MAX_CARRIES);
    carries[n_carries].n = n;
    carries[n_carries].d = d;
    n_carries++;
  } 
  qsort(carries, n_carries, sizeof(Rational), cmp_rational);

  if (strlen(carry_alphabet) != n_carries)
  {
    fprintf(stderr, "Carry alphabet wrong length (should be %d.)\n", n_carries);
    return 4;
  }

  printf("// OpenSCAD file providing tables for multiplier rods using Slonimsky's theorem.\n\n");
  printf("\n");
  printf(" base = %d;\n", base);
  printf(" ncarrysymbols = %d;\n", n_carries);
  printf("\n// Digit alphabet = \"%s\"\n", dig_alphabet);
  printf("// Carry alphabet = \"%s\"\n", carry_alphabet);
  printf("\n\n"); 

  printf("// Fractional breakpoints:\n");
  printf("// ");
  for (i = 0; i < n_carries; )
  {
    printf("%d/%d", carries[i].n, carries[i].d);
    i++;
    if (i < n_carries)
       printf(", ");
  }
  printf(".\n\n");

  printf("// Carry sequences:\n");
  for (i = 0; i < n_carries; i++)
  {
    printf("// (");
    for (j = 0; j < base; )
    {
      putchar(dig_alphabet[(carries[i].n * j) / carries[i].d]);
      j++;
      if (j < base)
         putchar(',');
    }
    printf(") from %d/%d to %d/%d", 
           carries[i].n, carries[i].d,
           (i == n_carries - 1 ? 1 : carries[i+1].n), 
           (i == n_carries - 1 ? 1 : carries[i+1].d));
    printf("\n");
  }
  printf("\n");

  printf("// Rod tables:\n\n");
  for (dig = 0; dig < base; dig++)
  {
    printf("// Multiplier digit: %d (symbol %c)\n\n", dig, dig_alphabet[dig]);
    printf("//   carry-in %s  carry-out\n\n", dig_alphabet); 
    for (i = 0; i < n_carries; i++)
    {
      printf("//      %c     ", carry_alphabet[i]);
      for (j = 0; j < base; j++)
          putchar(dig_alphabet[get_output_digit(dig, i, j)]);
      printf("     %c\n", carry_alphabet[get_cout(dig, i)]);
    }
    printf("\n\n"); 
  }


  printf("digits = [");
  for (dig = 0; dig < base; )
  {
    printf("%d", (int)dig_alphabet[dig]);
    dig++;
    if (dig < base)
       printf(", ");
  }
  printf("];\n\n");


  printf("carryins = [");
  for (i = 0; i < n_carries; )
  {
    printf("%d", (int)carry_alphabet[i]);
    i++;
    if (i < n_carries)
       printf(", ");
  }
  printf("];\n\n");


  printf("carryouts = [\n");
  for (dig = 0; dig < base; )
  {
    printf("  [");
    for (i = 0; i < n_carries; )
    {
      printf("%d", (int)carry_alphabet[get_cout(dig, i)]);
      i++;
      if (i < n_carries)
         printf(", ");
    }
    printf("]");
    dig++;
    if (dig < base)
       printf(",");
    printf("\n");
  }
  printf("];\n\n");
  
  printf("outputdigits = [\n");
  for (dig = 0; dig < base; )
  {
    printf("  [");
    for (i = 0; i < n_carries; )
    {
      printf("[");
      for (j = 0; j < base; )
      {
        printf("%d", (int)dig_alphabet[get_output_digit(dig, i, j)]);
        j++;
        if (j < base)
            printf(", ");
      }
      printf("]");
      i++;
      if (i < n_carries)
         printf(", \n   ");
    }
    printf("]");
    dig++;
    if (dig < base)
       printf(",");
    printf("\n\n");
  }
  printf("];\n\n");

  return 0;
}
