// Language: C++ 

//
// Renders a font written in a primitive font description language into 
// OpenSCAD polygons (and possibly PBM bitmaps.)
//

//
// The font description language used has not been completely implemented as 
// this program does not compute intersections between line segments and 
// circular arcs, or between circular arcs and circular arcs.  However, this 
// program suffices to compute a basic font.
//

// This file is in the public domain.

// David Moews, 3-VI-2014.

#include <cassert>
#include <cstdlib>
#include <cmath>

#include <utility>     // std::pair
#include <vector>
#include <string>
#include <iostream>
#include <fstream>
#include <sstream>

//
// POINTVEC
//
// Generic 2-dimensional point or vector.
//

typedef std::pair<double, double> PointVec;

PointVec operator +(const PointVec &a, const PointVec &b)
{
  return PointVec(a.first + b.first, a.second + b.second);
}

PointVec operator -(const PointVec &a, const PointVec &b)
{
  return PointVec(a.first - b.first, a.second - b.second);
}

PointVec operator *(double r, const PointVec &a)
{
  return PointVec(r * a.first, r * a.second);
}

double dot(const PointVec &a, const PointVec &b)
{
  return a.first * b.first + a.second * b.second;
}

double length(const PointVec &a)
{
  return std::sqrt(dot(a, a));
}

double dist(const PointVec &a, const PointVec &b)
{
  return length(a - b);
}

PointVec rotate_90_ccw(const PointVec &a)
{
  return PointVec(-a.second, a.first);
}

PointVec rotate_90_cw(const PointVec &a)
{
  return PointVec(a.second, -a.first);
}

PointVec normalize(const PointVec &a)
{
  double l = length(a);
  if (l == 0.0)
     return a;
  else
     return (1.0 / l) * a;
}


#define FUZZ 1e-5

//
// If lines ab and cd are nonparallel, returns true & fills in parameter values
// t and u of the intersection, which is at (1-t) a + t b = (1-u) c + u d.
// If lines are parallel or degenerate, returns false.
//
bool parametric_intersection(double *p_t, const PointVec &a, const PointVec &b,
                             double *p_u, const PointVec &c, const PointVec &d)
{
// Need to solve: (b-a) t + (c-d) u = c-a.
  PointVec v1 = b - a;
  PointVec v2 = c - d;
  PointVec w = c - a;
  double det = v1.first * v2.second - v1.second * v2.first;
  if (std::fabs(det) < FUZZ)
     return false;
  *p_t = (v2.second * w.first - v2.first * w.second) / det;
  *p_u = (- v1.second * w.first + v1.first * w.second) / det;
  return true;
}

//
//  POLYGON
//
//  Used to represent polylines or polygons.
//

typedef std::vector<PointVec> Polygon;

// Assumes p closed.
bool inside_poly(double x, double y, const Polygon &p)
{
  int num_intersections = 0;

  if (p.empty())
     return 0;

  if (p.size() == 1)
     return p.begin()->first == x && p.begin()->second == y;

  // Count intersections of ray from (x,y) to (x, +infinity) with polygon 
  for (std::size_t i = 0, j = 1; i < p.size(); i++, j++)
  {
    if (j == p.size())
       j = 0;

    double x1 = p[i].first;
    double y1 = p[i].second;

    if (x1 == x && y1 == y)
       return true;

    double x2 = p[j].first;
    double y2 = p[j].second;
    if (x1 > x2)
    {
      double temp = x1; x1 = x2; x2 = temp;
      temp = y1; y1 = y2; y2 = temp;
    }

    if (x1 == x && x2 == x &&
        ((y1 <= y && y2 >= y) || (y2 <= y && y1 >= y)))
       return true;
        
    if (x1 > x || x2 <= x || (y1 < y && y2 < y))
       continue;

    double y_line = ((x - x1) / (x2 - x1)) * (y2 - y1) + y1;
    if (y == y_line)
       return true;
    if (y < y_line)
       num_intersections++;
  }
  return (num_intersections % 2) == 1;  
}

//
// GRID
// 
// Used to represent a bitmap which is drawn into.
//

struct Grid
{
  double ofs_x, ofs_y;
  double scale;
  std::size_t pix_w, pix_h;
  std::vector<bool> pixmap;

  void clear()
  {
    pixmap = std::vector<bool>(pix_w * pix_h, false);
  }
};

// Assumes p closed.
void add_polygon_to_grid(Grid *p_g, const Polygon &p)
{
  for (std::size_t y = 0, yc = p_g->pix_h - 1; y < p_g->pix_h; y++, yc--)
  for (std::size_t x = 0; x < p_g->pix_w; x++)
      if (inside_poly(x / p_g->scale - p_g->ofs_x, y / p_g->scale - p_g->ofs_y, p))
          p_g->pixmap[x + yc * p_g->pix_w] = true;
}

// Assumes outer, hole closed.
void add_holed_polygon_to_grid(Grid *p_g, const Polygon &outer, const Polygon &hole)
{
  for (std::size_t y = 0, yc = p_g->pix_h - 1; y < p_g->pix_h; y++, yc--)
  for (std::size_t x = 0; x < p_g->pix_w; x++)
  {
    double x0 = x / p_g->scale - p_g->ofs_x;
    double y0 = y / p_g->scale - p_g->ofs_y;
    if (inside_poly(x0, y0, outer) && !inside_poly(x0, y0, hole))
        p_g->pixmap[x + yc * p_g->pix_w] = true;
  }
}

// Write grid to file named fn (PBM format.)
void dump(const char *fn, const Grid &grid)
{
  std::ofstream g(fn);
  g << "P1\n" << grid.pix_w << ' ' << grid.pix_h << '\n';
  for (std::vector<bool>::const_iterator i = grid.pixmap.begin(); i != grid.pixmap.end(); ++i)
      g << (*i ? "1 " : "0 ");
}

//
// OPENSCADSURFACE
//
// Used to accumulate polygons for output in OpenSCAD format.
//

struct OpenSCADPolygon
{
  Polygon outer, inner;

  OpenSCADPolygon(const Polygon &o) : outer(o), inner() { }
  OpenSCADPolygon(const Polygon &o, const Polygon &c) : outer(o), inner(c) { }
};

typedef std::vector<OpenSCADPolygon> OpenSCADSurface;

// Assumes p closed.
void add_polygon_to_openscad(OpenSCADSurface *p_o, const Polygon &p)
{
  p_o->push_back(OpenSCADPolygon(p));
}

// Assumes outer, hole closed.
void add_holed_polygon_to_openscad(OpenSCADSurface *p_o, const Polygon &outer, const Polygon &hole)
{
  p_o->push_back(OpenSCADPolygon(outer, hole));
}

void openscad_write_points(std::ostream &o, const Polygon &p)
{
  for (Polygon::const_iterator j = p.begin();;)
  {
    o << "[" << j->first << "," << j->second << "]";
    if (++j == p.end())
       break;
    o << ", ";
  }
}

// Write sequence st, st+1, ..., end-2, end-1.
void openscad_write_sequence(std::ostream &o, std::size_t st, std::size_t end)
{
  for (std::size_t j = st;;)
  {
    o << j;
    if (++j == end)
       break;
    o << ", ";
  }
}

// Write OpenSCAD surface for character c to std::cout, as module.
void dump_openscad(char c, const OpenSCADSurface &o)
{
  std::cout << "module character" << (int)c << "()\n{\n";
  for (OpenSCADSurface::const_iterator i = o.begin(); i != o.end(); ++i)
  {
    std::cout << "  polygon(points = [";
    openscad_write_points(std::cout, i->outer);
    if (!i->inner.empty())
    {
      std::cout << ", ";
      openscad_write_points(std::cout, i->inner);
      std::cout << "], paths = [[";
      openscad_write_sequence(std::cout, 0, i->outer.size());
      std::cout << "], [";
      openscad_write_sequence(std::cout, i->outer.size(), i->outer.size() + i->inner.size());
      std::cout << "]";
    } 
    std::cout << "]);\n";
  }
  std::cout << "};\n\n\n";
}


//
// STROKE
//
//  A portion of a drawing path.  May be straight or a circular arc.
//


// Segment from st to fin; or, circular arc from st to fin with center ctr, 
// traversed in either the clockwise (is_ccww == false) 
// or counterclockwise (is_ccww == true) direction.
class Stroke
{
  bool is_circ;
  bool is_ccww;
  PointVec st;
  PointVec fin;
  PointVec ctr;

public:

  Stroke(const PointVec &c, const PointVec &d) 
                           : is_circ(false), is_ccww(false), st(c), fin(d) { }
  Stroke(const PointVec &c, const PointVec &d, const PointVec &ctr0, bool ccw) 
                           : is_circ(true), is_ccww(ccw), st(c), fin(d), ctr(ctr0) { }

  PointVec start() const { return st; } 
  PointVec end() const { return fin; } 

  void replace_start(const PointVec &c) { st = c; }
  void replace_end(const PointVec &c) { fin = c; }

  PointVec center() const 
  { 
    if (!is_circ)
       return PointVec();
    else
       return ctr; 
  } 
  bool is_circle() const { return is_circ; }
  bool is_ccw() const 
  { 
    if (is_circ) 
       return is_ccww; 
    else 
       return false; 
  }
    
  PointVec start_dir() const
  {
    if (!is_circ)
       return normalize(fin - st);
    else if (is_ccww)
       return normalize(rotate_90_ccw(st - ctr));
    else 
       return normalize(rotate_90_cw(st - ctr));
  }
  PointVec end_dir() const
  {
    if (!is_circ)
       return normalize(fin - st);
    else if (is_ccww)
       return normalize(rotate_90_ccw(fin - ctr));
    else 
       return normalize(rotate_90_cw(fin - ctr));
  }
  Stroke reverse() const
  {
    if (!is_circ)
       return Stroke(fin, st);
    else 
       return Stroke(fin, st, ctr, !is_ccww);
  }

  // Shift strokd perpendicular to start and end directions; 
  // positive amt is to the right, as seen by a walker along the path; 
  // negative is left.
  // Returns <true, resultant stroke> on success,
  // <false, arbitrary stroke> otherwise.
  std::pair<bool, Stroke> shift_by(double amt) const
  {
    if (!is_circ)
    {
      PointVec shift = amt * rotate_90_cw(start_dir());
      return std::pair<bool, Stroke>(true, Stroke(st + shift, fin + shift));
    }
    double rad1 = dist(st, ctr);
    double rad2 = dist(fin, ctr);
    PointVec rhat1 = normalize(st - ctr);
    PointVec rhat2 = normalize(fin - ctr);
    if (is_ccww)
       amt = -amt;
    if (amt > rad1 || amt > rad2)
       return std::pair<bool, Stroke>(false, Stroke(PointVec(), PointVec()));
    return std::pair<bool, Stroke>(true, Stroke(st - amt * rhat1, fin - amt * rhat2, ctr, is_ccww));
  }

  // Returns intersections in the order seen by a walker along *this.
  // First intersection point (if existent) is placed in *p_i1;
  // second (if existent) is placed in *p_i2.
  int num_intersections(const Stroke &p, PointVec *p_i1, PointVec *p_i2)
  {
    if (!is_circ && !p.is_circ)
    {
      double t, u;
      if (!parametric_intersection(&t, start(), end(), &u, p.start(), p.end()))
         return 0;
      if (t < 0.0 || t > 1.0 || u < 0.0 || u > 1.0)
         return 0;
      *p_i1 = (1.0 - t) * start() + t * end();
      return 1;
    }
    // Intersection of circular arcs with segments or arcs with arcs
    // has not yet been implemented, so we just check to see if the given 
    // input segments butt exactly against each other, and abort otherwise.
    if (dist(end(), p.start()) > FUZZ)
    {
      std::cerr << "Strokes do not match, and intersection with arcs is not yet implemented.  Stop.\n";
      std::exit(2);
    } 
    return 0;
  }
};

//
// RENDERPARMS
//

struct RenderParms
{
  double stroke_halfwidth;
  bool extend_left;
  bool extend_right;
  int points_per_arc;
};

//
// PATH
//
//   Used for both open & closed paths.
//

typedef std::vector<Stroke> Path;

// Compute polygonized version of input path and return it.
Polygon polygonize_path(const RenderParms &rp, const Path &path)
{
  if (path.empty())
     return Polygon();

  Polygon rv(1, path.begin()->start());

  for (Path::const_iterator i = path.begin(); i != path.end(); ++i)
  if (!i->is_circle())
     rv.push_back(i->end());
  else 
  {
    assert(rp.points_per_arc > 0);
    double rad1 = dist(i->start(), i->center());
    double rad2 = dist(i->end(), i->center());
    PointVec rhat1 = normalize(i->start() - i->center());  
    PointVec rhat2 = normalize(i->end() - i->center()); 
    PointVec tangent1 = i->is_ccw() ? rotate_90_ccw(rhat1) : rotate_90_cw(rhat1);
    double total_angle = std::atan2(dot(rhat2, tangent1), dot(rhat2, rhat1));
    if (total_angle < 0.0)
       total_angle += 2.0 * 3.141592653589793238;
    double angular_step = total_angle / rp.points_per_arc;
    double radius_step = (rad2 - rad1) / rp.points_per_arc;
    double r = rad1, ang = 0.0;
    for (int j = 0; j < rp.points_per_arc; j++)
    {
      r += radius_step;
      ang += angular_step;
      rv.push_back(PointVec(i->center() + (r * std::cos(ang)) * rhat1
                                        + (r * std::sin(ang)) * tangent1));
    }
  }
  return rv;
}


// Fill in a gap between p1 (approached with direction dir1) 
// and p2 (departed from with direction dir2) by appending segments to the 
// input path, *p_path, which is assumed to end at p1.
//
// Uses miter for a right-angle joint and bevel otherwise.
// 
void fill_in_path(const RenderParms &rp, Path *p_path, 
                  const PointVec &p1, const PointVec &dir1, const PointVec &p2, const PointVec &dir2)
{
  if (std::fabs(dot(dir1, dir2)) < FUZZ)
  {
    double t, u;
    assert(parametric_intersection(&t, p1, p1 + dir1, &u, p2, p2 + dir2));
    p_path->push_back(Stroke(p1, p1 + t * dir1));
    p_path->push_back(Stroke(p2 + u * dir2, p2));
  } else
    p_path->push_back(Stroke(p1, p2));
}

// Add stroke to end of path *p_output_path, intersecting or filling in as 
// necessary.  Only deals with intersection of the new stroke with the 
// immediately preceding stroke.
void add_stroke(const RenderParms &rp, Path *p_output_path, const Stroke &stroke)
{
  if (p_output_path->empty())
  {
    p_output_path->push_back(stroke);
    return;
  }

  PointVec p1, p2;
  int i = (p_output_path->end() - 1)->num_intersections(stroke, &p1, &p2);
  if (i > 0)
  {
    (p_output_path->end() - 1)->replace_end(p1);
    p_output_path->push_back(stroke);
    (p_output_path->end() - 1)->replace_start(p1);
  } else
  {
    if (dist(stroke.start(), (p_output_path->end() - 1)->end()) > FUZZ)
    {
      fill_in_path(rp, p_output_path, 
                       (p_output_path->end() - 1)->end(), (p_output_path->end() - 1)->end_dir(),
                       stroke.start(), stroke.start_dir());
    }
    p_output_path->push_back(stroke);
  }
}

// Make path *p_path closed.  Only deals with intersection if it is between 
// the start and end stroke.
void close_path(const RenderParms &rp, Path *p_path)
{
  if (p_path->size() <= 1)
     return;

  PointVec p1, p2;
  int i = (p_path->end() - 1)->num_intersections(*p_path->begin(), &p1, &p2);
  if (i > 0)
  {
    (p_path->end() - 1)->replace_end(p1);
    p_path->begin()->replace_start(p1);
  } else
  if (dist(p_path->begin()->start(), (p_path->end() - 1)->end()) > FUZZ)
     fill_in_path(rp, p_path, (p_path->end() - 1)->end(), (p_path->end() - 1)->end_dir(),
                               p_path->begin()->start(), p_path->begin()->start_dir());
}

#undef FUZZ

// Given input open path, compute the region given by making a finite width 
// stroke along that path, and return a closed path which is the boundary of 
// that region,
Path render_stroke(const RenderParms &rp, const Path &in_path)
{
  if (in_path.empty())
     return Path();

  Path path = in_path;
  Path out_path;
  if (rp.extend_left)
  {
    PointVec p = in_path[0].start();
    PointVec p_dir = in_path[0].start_dir();
    path.insert(path.begin(), Stroke(p - rp.stroke_halfwidth * p_dir, p));
  }
  if (rp.extend_right)
  {
    PointVec q = (in_path.end() - 1)->end();
    PointVec q_dir = (in_path.end() - 1)->end_dir();
    path.push_back(Stroke(q, q + rp.stroke_halfwidth * q_dir));
  }

  PointVec p_st = path.begin()->start();
  PointVec shift_st = rotate_90_cw(path.begin()->start_dir());
  add_stroke(rp, &out_path, 
             Stroke(p_st + (-rp.stroke_halfwidth) * shift_st, p_st + rp.stroke_halfwidth * shift_st));
  for (Path::const_iterator i = path.begin(); i != path.end(); ++i)
  {
    std::pair<bool, Stroke> shifted_stroke = i->shift_by(rp.stroke_halfwidth);
    if (shifted_stroke.first)
       add_stroke(rp, &out_path, shifted_stroke.second);
  }
  PointVec p_end = (path.end() - 1)->end();
  PointVec shift_end = rotate_90_cw((path.end() - 1)->end_dir());
  add_stroke(rp, &out_path,
              Stroke(p_end + rp.stroke_halfwidth * shift_end, p_end + (-rp.stroke_halfwidth) * shift_end));
  for (Path::const_iterator i = path.end(); i != path.begin(); --i)
  {
    std::pair<bool, Stroke> shifted_stroke = (i-1)->shift_by(-rp.stroke_halfwidth);
    if (shifted_stroke.first)
       add_stroke(rp, &out_path, shifted_stroke.second.reverse());
  }
  close_path(rp, &out_path);
  return out_path;
}

// Given input closed path, compute the region given by making a finite 
// width stroke along that path, and return a pair of paths which are the 
// boundary of that region.
//
// The outer boundary is placed in *p_out_outer,
// and the inner boundary is placed in *p_out_hole.
//
// The input closed path should be traversed in a counterclockwise direction.
void render_closed_stroke(const RenderParms &rp, const Path &path, Path *p_out_outer, Path *p_out_hole)
{
  if (path.empty())
  {
    *p_out_outer = Path();
    *p_out_hole = Path();
    return;
  }

  *p_out_outer = Path();
  for (Path::const_iterator i = path.begin(); i != path.end(); ++i)
  {
    std::pair<bool, Stroke> shifted_stroke = i->shift_by(rp.stroke_halfwidth);
    if (shifted_stroke.first)
       add_stroke(rp, p_out_outer, shifted_stroke.second);
  }
  close_path(rp, p_out_outer);

  *p_out_hole = Path();
  for (Path::const_iterator i = path.end(); i != path.begin(); --i)
  {
    std::pair<bool, Stroke> shifted_stroke = (i-1)->shift_by(-rp.stroke_halfwidth);
    if (shifted_stroke.first)
       add_stroke(rp, p_out_hole, shifted_stroke.second.reverse());
  }
  close_path(rp, p_out_hole);
}

//
// MAIN PROGRAM
//

void usage(const char *c)
{
  std::cerr << "Renders description of a font written in a primitive font description language\n"
            << "(given on standard input) into OpenSCAD modules (written on standard output.)\n"
            << '\n'
            << "Usage: " << c << " stroke-halfwidth points-per-arc [pbm-file-prefix]\n"
            << "\n"
            << "where\n"
            << "\n"
            << "stroke-halfwidth        is half the width of each stroke.\n"
            << "points-per-arc          is the number of points used to render each arc.\n"
            << "pbm-file-prefix         If this argument is present, the program also writes\n"
            << "                        PBM files displaying each character in the font;\n"
            << "                        this prefix is used as the first portion of the\n"
            << "                        name of the PBM files written.\n";
}
  
//
// std::cin should be the input font description file.
// The OpenSCAD font is written to std::cout.
// If a PBM file prefix is given, the program will also
// create PBM files which are renderings of the characters in the font.
//
int main(int argc, char **argv)
{
  Grid the_grid;
  RenderParms the_render_parms;
  OpenSCADSurface the_openscad;
  int points_per_arc;
  double stroke_halfwidth;
  bool use_grid;
  const char *output_prefix;
  std::vector<char> characters_included;

  if ((argc < 3) || (argc > 4)
                 || ((stroke_halfwidth = std::atof(argv[1])) <= 0)
                 || ((points_per_arc = std::atoi(argv[2])) <= 0))
  {
    usage(argv[0]);
    return 1;
  }

  if ((use_grid = (argc == 4)))
     output_prefix = argv[3];
  else
     output_prefix = NULL;

  if (use_grid)
  {
    the_grid.ofs_x = 1.0;
    the_grid.ofs_y = 1.0;
    the_grid.scale = 100.0;
    the_grid.pix_w = 600;
    the_grid.pix_h = 800;
  }

  the_render_parms.stroke_halfwidth = stroke_halfwidth;
  the_render_parms.points_per_arc = points_per_arc;
  the_render_parms.extend_left = false;
  the_render_parms.extend_right = false;

  char currently_rendering;
  bool are_rendering = false;

  std::cout << "// OpenSCAD file, containing two-dimensional outlines of characters\n";
  std::cout << "//                in a simple font.\n\n";

  for (;;)
  {
    std::string l;

    if (!std::getline(std::cin, l))
       break;

    if (l.empty())
       continue;     // Ignore

    switch(l[0])
    {
      case 'F':
          if (l.size() < 3)
             continue;       // Ignore

          if (are_rendering)
          {
            if (use_grid)
               dump((std::string(output_prefix) + currently_rendering).c_str(), the_grid);
            dump_openscad(currently_rendering, the_openscad);
            the_openscad = OpenSCADSurface();
          }

          if (use_grid)
              the_grid.clear();
          are_rendering = true;
          currently_rendering = l[2]; 
          characters_included.push_back(currently_rendering);
          break;

      case 'O':
      case 'P':
          if (!are_rendering)
             continue;

          {
            std::istringstream line(l.substr(2));
  
            double x0, y0, x1, y1, xc, yc;
            char cw_ccw;
  
            line >> x0 >> y0;
            Path p;
  
            char c;

            while (line >> c)
            {
              switch (c)
              {
                case 'L':
                      line >> x1 >> y1;   
                      p.push_back(Stroke(PointVec(x0, y0), PointVec(x1, y1)));
                      x0 = x1;
                      y0 = y1;
                      break;
  
                case 'C':
                      line >> x1 >> y1 >> xc >> yc >> cw_ccw;    
                      p.push_back(Stroke(PointVec(x0, y0), PointVec(x1, y1), PointVec(xc, yc), cw_ccw == '-'));
                      x0 = x1;
                      y0 = y1;
                      break;
  
                default:
                      // Ignore
                      break;
              }
            }
            if (l[0] == 'P')
            {
              Path p_outside, p_inside;
              render_closed_stroke(the_render_parms, p, &p_outside, &p_inside);
              Polygon poly_outside = polygonize_path(the_render_parms, p_outside);
              Polygon poly_inside = polygonize_path(the_render_parms, p_inside);
              if (use_grid)
                 add_holed_polygon_to_grid(&the_grid, poly_outside, poly_inside);
              add_holed_polygon_to_openscad(&the_openscad, poly_outside, poly_inside);
            } else
            {
              Path stroked_p = render_stroke(the_render_parms, p);
              Polygon stroked_poly = polygonize_path(the_render_parms, stroked_p);
              if (use_grid)
                 add_polygon_to_grid(&the_grid, stroked_poly);
              add_polygon_to_openscad(&the_openscad, stroked_poly);
            }
          }
          break;
 
      case '#':
          // Comment
          break;

      default:
          // Ignore 
          break;
    }
  }

  if (are_rendering)
  {
    if (use_grid)
       dump((std::string(output_prefix) + currently_rendering).c_str(), the_grid);
    dump_openscad(currently_rendering, the_openscad);
  }

  std::cout << "module character(c)\n{\n";

  for (std::vector<char>::const_iterator p = characters_included.begin();
       p != characters_included.end(); ++p)
  {
    std::cout << "  if (c == " << (int)*p << ") { character" << (int)*p << "(); }\n  else\n";
  }
  std::cout << "  { }\n};\n\n";

  // Define function charwidth(c, h), which returns the width of character c 
  // when scaled to height h, and the module drawchar(c, h), which
  // draws character c scaled to height h in the (x,y)-plane,
  // with its approximate lower left corner at (0,0).
  //
  // Assumes a 4 x 6 bounding box for characters in the font.
  std::cout 
<< "function charwidth(c, h) = 4 * h / 6;\n\n"
<< "module drawchar(c, h)\n{\n"
<< "  scale(v = [h / 6, h / 6, 0])\n"
<< "     character(c);\n"
<< "};\n";

  return 0;
}
