// OpenSCAD file to construct multiplier rods, cradle for rods, and 
// lid for cradle; the rods use Slonimsky's theorem.

// Must be preceded by definitions of the variables base, ncarrysymbols,
// digits, carryins, carryouts, and outputdigits, the function
// charwidth(c, h), which returns the width of character c when
// scaled to height h, and the module drawchar(c, h), which
// draws character c scaled to height h in the (x,y)-plane,
// with its approximate lower left corner at (0,0).
//
//  David Moews, 22-VI-2014.
//  
//  This file is in the public domain.



// Basic dimensional definitions

// Radius of fundamental polygon
r = 24;

// Height of digit on top of rod
htop = 34;

// Height of digits on side of rod which replicate digit on top of rod
hsides = 6;

// Height of carry symbols on side of rod
hcarry = 4;

// Height of output digits on side of rod
houtput = 4;

// Axle height
axleht = 10;

// Top axle radius
rtopaxle = 5;

// Bottom axle radius
rbotaxle = 8;

// Knob height
knobht = 10;

// Depth of all embossed characters
// (half raised above surface, half buried below)
chardepth = 1;

// Clearance between different parts 
clearance = 1.0 + chardepth / 2.0;

// Extra horizontal clearance between adjacent rods
horizclearance = 1.0 + chardepth / 2.0;

// Clearance around axle and tabs
axleclearance = 0.5;

// Lid height
lidheight = 2.0;

// Slot width expansion factor for lid
lidslotwidthmultiplier = 1.3;


// Further dimensions computed from the above


// Apothem of fundamental polygon
apothem = r * cos(180 / ncarrysymbols);

// Side of fundamental polygon 
side = 2 * r * sin(180 / ncarrysymbols);

// Total height of rod 
totheight = hsides * 2 + hcarry * 6 + houtput * 1.2 * base;

// Cradle wall thickness
wall = axleht - 2 * clearance;

// Lid slot width
slotwidth = side * lidslotwidthmultiplier;

// Radius of hole in cradle for top axle
rtophole = rtopaxle + axleclearance;

// Radius of hole in cradle for bottom axle
rbothole = rbotaxle + axleclearance;

// Inside dimensions of cradle
insidewidth = totheight + 2 * clearance;
insidedepth = 2 * r + 2 * clearance + axleclearance;
function insidelength(numrods) = 
          horizclearance * (numrods - 1) + clearance * (numrods + 1) + 2 * r * numrods;

// X coordinate of center of rod i in cradle
function rodxcoord(i) = horizclearance * i + clearance * (i + 1) + r * (2*i + 1);


// Modules to construct a rod


module polygonalrod(rr, hh, ss)
{
   linear_extrude(height = hh)
   {
     for (i = [0 : ss - 1])
     {
       polygon(points = [[0, 0], [rr * cos(180 * (2*i+1) / ss), rr * sin(180 * (2*i+1) / ss)],
                                 [rr * cos(180 * (2*i-1) / ss), rr * sin(180 * (2*i-1) / ss)]]);
     }
   }
};

module basicrod()
{
  rotate(a = [-90, -90, 0])
  union()
  {
    polygonalrod(r, totheight, ncarrysymbols);

    translate(v = [0, 0, - axleht])
        cylinder(r = rbotaxle, h = axleht);

    translate(v = [0, 0, totheight])
        cylinder(r = rtopaxle, h = axleht);

    translate(v = [0, 0, - axleht - knobht])
        cylinder(r = r, h = knobht);

    translate(v = [0, 0, axleht + totheight])
        cylinder(r = r, h = knobht);
  }
};

module embossedtext(dig)
{
  rotate(a = [-90, -90, 0])
  union()
  {
    // Character on top of cylinder
    translate(v = [-charwidth(digits[dig], htop) / 2, -htop / 2, axleht + knobht + totheight - chardepth / 2])
         linear_extrude(height = chardepth)
             drawchar(digits[dig], htop);

    for (i = [0 : ncarrysymbols - 1])
    {
      rotate(a = [0, 0, 360 * i / ncarrysymbols])
         translate(v = [apothem, 0, 0])
            rotate(a = [90, 0, 90])
            {
              translate(v = [-charwidth(digits[dig], hsides) / 2, totheight - 2 * hsides, - chardepth / 2])
                   linear_extrude(height = chardepth)
                       drawchar(digits[dig], hsides);

              translate(v = [-charwidth(carryins[i], hcarry) / 2, totheight - 2 * hsides - 2 * hcarry, - chardepth / 2])
                   linear_extrude(height = chardepth)
                       drawchar(carryins[i], hcarry);

              translate(v = [-charwidth(carryouts[dig][i], hcarry) / 2, hcarry, - chardepth / 2])
                   linear_extrude(height = chardepth)
                       drawchar(carryouts[dig][i], hcarry);

                 for (j = [0 : base - 1])
                 {
                   translate(v = [-charwidth(outputdigits[dig][i][j], houtput) / 2, 
                                  3 * hcarry + (base - 1 - j) * 1.2 * houtput,
                                  - chardepth / 2])
                       linear_extrude(height = chardepth)
                            drawchar(outputdigits[dig][i][j], houtput);
                 }
            }
    }
  }
};



// The rod for digit d is the union of basicrod() and embossedtext(d).
// Alternatively, contrasting colors could be used for
// embossedtext(d) and the difference of basicrod() and embossedtext(d).



// Modules to construct the cradle and lid.
// The argument numrods should be set to the number of rods that the cradle 
// should hold.

// Tabs to be placed on top of the cradle, or subtracted from the lid.
// c is a clearance value.
module tabs(numrods, c)
{
  union()
  {
    translate(v = [- wall - c, 2*wall - c, insidedepth - c])
         cube(size = [wall + 2*c, 2.5*wall + 2*c, lidheight + wall + 2*c]);
    translate(v = [insidelength(numrods) - c, insidewidth - 6*wall - c, insidedepth - c])
         cube(size = [wall + 2*c, 4*wall + 2*c, lidheight + wall + 2*c]);
  }
};

module cradle(numrods)
{
  difference()
  {
    union()
    {
       translate(v = [-wall, -wall, -wall])
            cube(size = [insidelength(numrods) + 2*wall, insidewidth + 2*wall, insidedepth + wall]);
       tabs(numrods, 0);
    }
    union()
    {
       cube(size = [insidelength(numrods), insidewidth, insidedepth + clearance]);
       for (i = [0 : numrods - 1])
       {
         translate(v = [rodxcoord(i), clearance, r + clearance + axleclearance])
           rotate(a = [90, 0, 0])
             union()
             {
               translate(v = [-rbothole, 0, 0])
                   cube(size = [2 * rbothole, r + 2 * clearance, wall + 2 * clearance]);
               cylinder(h = wall + 2 * clearance, r = rbothole);
             }

         translate(v = [rodxcoord(i), insidewidth + wall + clearance, r + clearance + axleclearance])
           rotate(a = [90, 0, 0])
             union()
             {
               translate(v = [-rtophole, 0, 0])
                   cube(size = [2 * rtophole, r + 2 * clearance, wall + 2 * clearance]);
               cylinder(h = wall + 2 * clearance, r = rtophole);
             }
       }
    }
  }
};

module basiclid(numrods)
{
  difference()
  {
    translate(v = [- 2*wall, - wall, insidedepth])
       cube(size = [insidelength(numrods) + 4*wall, insidewidth + 2*wall, lidheight]);
    union()
    {
       tabs(numrods, axleclearance);
       for (i = [0 : numrods - 1])
       {
         translate(v = [rodxcoord(i) - slotwidth / 2.0, clearance, insidedepth - clearance])
             cube(size = [slotwidth, totheight, lidheight + 2*clearance]);
       }
    }
  }
};


module embossedlidtext(numrods)
{
  union()
  {
    for (i = [0 : numrods - 1])
    {
      if (i < numrods - 1)
      {
        // Guide lines from carry-in to carry-out 
        translate(v = [rodxcoord(i) + slotwidth / 2.0, 
           clearance + totheight - 2 * hsides - 2 * hcarry, 
           insidedepth + lidheight - chardepth / 2.0])
           cube(size = [(2 * r + clearance + horizclearance - slotwidth) / 2.0, hcarry, chardepth]);
    
        translate(v = [rodxcoord(i) + horizclearance * 0.5 + clearance * 0.5 + r,
           clearance + hcarry,
           insidedepth + lidheight - chardepth / 2.0])
           cube(size = [(2 * r + clearance  + horizclearance - slotwidth) / 2.0, hcarry, chardepth]);
    
        translate(v = [rodxcoord(i) + horizclearance * 0.5 + clearance * 0.5 + r - hcarry / 2.0,
           clearance + hcarry,
           insidedepth + lidheight - chardepth / 2.0])
           cube(size = [hcarry, totheight - 2 * hsides - 2 * hcarry, chardepth]);
      }

      // Embossed digits on lid
      for (j = [0 : base - 1])
      {
        translate(v = [rodxcoord(i) - clearance - slotwidth / 2.0 - charwidth(digits[j], houtput), 
                       3 * hcarry + (base - 1 - j) * 1.2 * houtput + clearance, 
                       insidedepth + lidheight - chardepth / 2.0])
                    linear_extrude(height = chardepth)
                          drawchar(digits[j], houtput); 

      }
    }
  }
};

// The cradle is cradle(numrods).

// The lid is the union of basiclid(numrods) and embossedlidtext(numrods).
// Alternatively, contrasting colors could be used for
// embossedlidtext(numrods) and the difference of 
// basiclid(numrods) and embossedlidtext(numrods).
